-- XRechnung über XML-Datei importieren
CREATE OR REPLACE FUNCTION x_10_interfaces.xrechnung__import__from_xml_bytea(
    IN xml_file bytea,
    IN doktype  varchar DEFAULT 'xrechnung',
    OUT dokunr  varchar,
    OUT dbrid   varchar
    )
    RETURNS record
    AS $$

      import sys
      import re

      # Relativer Pfad zu Modulen (ausgehend vom Arbeitsverzeichnis 'data')
      relative_path = '../python-plpythonu/Lib/site-packages'
      # Pfad nur hinzufügen, wenn er noch nicht in sys.path ist
      if relative_path not in sys.path:
        sys.path.append(relative_path)

      from drafthorse.models.document import Document
      from lxml import etree
      from io import BytesIO

      def safe_getattr(context, attr_path):
          """Holt ein Attribut basierend auf einem vollständigen Pfad, der das Objekt und das Attribut angibt."""
          try:
              parts = attr_path.split('.')
              obj = context.get(parts[0], None)
              for part in parts[1:]:
                  if part.startswith('children['):  # Zugriff auf ein Kind-Element
                      try:
                          index = int(part[9:-1])  # Extrahiert den Index, z.B. aus 'children[0]'
                          obj = obj.children[index]
                      except (IndexError, ValueError, AttributeError):
                          return None
                  elif hasattr(obj, part):  # Normaler Attributzugriff
                      obj = getattr(obj, part)
                  else:
                      return None
              return obj
          except Exception as e:
              plpy.error(f'Fehler beim Zugriff auf Attribut {attr_path}: {str(e)}')

      def extract_value(element):
          """Extrahiert den Wert basierend auf dem Typ des Elements."""
          try:
              if element is None:
                  return None
              elif element.__class__.__name__ == 'StringElement':
                  return element._text
              elif element.__class__.__name__ == 'DateTimeElement':
                  return str(element._value)
              elif element.__class__.__name__ == 'Decimal':
                  return str(element)
              elif element.__class__.__name__ == 'str':
                  return element
              elif element.__class__.__name__ == 'IndicatorElement':
                  return element
              return element
          except Exception as e:
              plpy.error(f'Fehler beim Extrahieren des Wertes: {str(e)}')

      def process_mapping_row(row, context):
          """Verarbeitet eine Mapping-Zeile und extrahiert Werte aus dem gegebenen Kontext."""

          values = []

          # Verarbeitung der 'erechmap_parameter' und Extraktion der Werte
          try:
              params = row['erechmap_parameter'].split(',')
              for param in params:
                  field = safe_getattr(context, param.strip())
                  value = extract_value(field)
                  values.append(value)
          except KeyError:
              # Fehlerbehandlung, wenn 'erechmap_parameter' fehlt
              plpy.error(f"Fehler: 'erechmap_parameter' fehlt in der Zeile erechmap_id: {row['erechmap_id']}")

          # Verarbeitung von 'erechmap_rewrite'
          if row['erechmap_rewrite']:
              matches = re.findall(r'\$([0-9]+)', row['erechmap_rewrite'])                        # Extrahieren der Parameternummern
              num_params = len(set(matches))                                                      # Anzahl der unterschiedlichen Parameternummern
              query = plpy.prepare(row['erechmap_rewrite'], ['text' for _ in range(num_params)])  # Erstellen der SQL-Abfrage
              result = plpy.execute(query, values)                                                # Ausführen der SQL-Abfrage
              # Überprüfen, ob das Ergebnis nicht leer ist
              if not result or len(result) == 0:
                  plpy.error(f"Fehler: Die Abfrage '{row['erechmap_rewrite']}' hat kein Ergebnis zurückgegeben. Parameterwerte: {values}. Das Feld {row['erechmap_column']} kann nicht gefüllt werden.")
              value = str(result[0][list(result[0].keys())[0]])                                   # Extrahieren des Wertes aus dem Ergebnis
          elif values:
              value = values[0]

          return row['erechmap_column'], value

      def process_mapping_table(context, cursor_query, fields, values):
          """Verarbeitet eine Mapping-Tabelle und extrahiert Werte aus dem gegebenen Kontext."""
          # Starte den Cursor für die Abfrage
          cursor = plpy.cursor(cursor_query)
          for row in cursor:
              try:
                  # Überprüfen und verarbeiten von erechmap_parameter
                  if row['erechmap_parameter']:
                      column, value = process_mapping_row(row, context)
                      if value is not None:
                          fields.append(column)
                          values.append("'" + str(value).replace("'", "''") + "'")

                  # per SQL gefüllte Felder übernehmen
                  if row['erechmap_sql']:
                      feld = plpy.execute(row['erechmap_sql'])
                      if feld:
                          first_key = list(feld[0].keys())[0]
                          if feld[0][first_key] is not None:
                              fields.append(row['erechmap_column'])
                              values.append("'" + str(feld[0][first_key]).replace("'", "''") + "'")

                  # per Konstante gefüllte Felder übernehmen
                  if row['erechmap_const']:
                      fields.append(row['erechmap_column'])
                      values.append("'" + str(row['erechmap_const']).replace("'", "''") + "'")

              except Exception as row_exception:
                  # Fehlerbehandlung auf Zeilenebene - Mehr Details zur betroffenen Zeile und Spalte
                  plpy.error(f"Fehler in der Verarbeitung von row mit erechmap_column='{row.get('erechmap_column', 'N/A')}', erechmap_parameter='{row.get('erechmap_parameter', 'N/A')}' "
                            f"und erechmap_sql='{row.get('erechmap_sql', 'N/A')}': {str(row_exception)}")

          return fields, values

      def remove_xml_comments_as_bytes(xml_bytes):
          """Entfernt Kommentare innerhalb der XML"""
          try:
              parser = etree.XMLParser(remove_comments=True)
              tree = etree.parse(BytesIO(xml_bytes), parser)
              return etree.tostring(tree.getroot())
          except Exception as e:
              plpy.error(f'Fehler beim Entfernen von XML-Kommentaren: {str(e)}')

      # Eingabedaten prüfen
      if xml_file is None:
          raise ValueError("Es wurde kein XRechnungs-XML übergeben.")

      # XML Parsen
      try:
          xml_content = remove_xml_comments_as_bytes(xml_file)
          doc = Document.parse(xml_content)
      except Exception as xml_parsing_error:
          plpy.error(f"Fehler beim Parsen des XML-Dokuments: {str(xml_parsing_error)}")

      # Dictionary befüllen, um Funktionen Kontextabhängig für Kopf-, Positions-, Zuschlagsdaten, ... ausführen zu können
      context = {'doc': doc}

      # Lieferantenkürzel finden
      try:
          seller_name = doc.trade.agreement.seller.name._text
          seller_kreditorennummer = doc.trade.agreement.seller.id._text

          # Falls seller_kreditorennummer leer oder None ist, ersetze sie durch None
          if not seller_kreditorennummer or seller_kreditorennummer.strip() == "":
              seller_kreditorennummer = None
          else:
              seller_kreditorennummer = int(seller_kreditorennummer)  # Konvertiere zu Integer, falls nicht leer

          query = """SELECT coalesce(
                         (SELECT a2_krz FROM adk2 WHERE a2_knr = $1 AND $1 IS NOT null LIMIT 1),    -- Lieferant über Kreditorennummer finden
                         (SELECT ad_krz FROM adk  WHERE ad_fa1 iLIKE $2 LIMIT 1)                    -- Lieferant über Name finden
                       ) AS ad_krz
                  """
          plan = plpy.prepare( query, ["integer", "varchar" ] )
          beld_krzlieferung = plpy.execute( plan, [seller_kreditorennummer, seller_name] )

          if not beld_krzlieferung:
              plpy.error(f"Zu der Rechnung wurde kein Lieferant gefunden: {seller_name} | {seller_kreditorennummer}")
      except Exception as supplier_error:
          plpy.error(f"Fehler beim Abrufen des Lieferantenkürzels für {seller_name}: {str(supplier_error)}")

      beld_krzlieferung = beld_krzlieferung[0]["ad_krz"]
      insert_fields = ['beld_krzlieferung']
      insert_values = ["'" + beld_krzlieferung + "'"]


      #############
      # Kopfdaten #
      #############

      # Auszulesende Felder für Kopfdaten aus Mappingtabelle holen
      try:
          cursor_query = f"""
            SELECT * FROM (
              SELECT *,
                     ROW_NUMBER() OVER (PARTITION BY erechmap_table, erechmap_column
                       ORDER BY CASE WHEN erechmap_ad_krz = '{beld_krzlieferung}' THEN 0 ELSE 1 END) AS rn
                FROM x_10_interfaces.e_rechnung_mapping
               WHERE erechmap_table = 'eingrechdokument'
                 AND erechmap_doktype = '{doktype}'
                 AND (erechmap_ad_krz = '{beld_krzlieferung}' OR erechmap_ad_krz IS NULL)
              ) sub
              WHERE rn = 1
            """
          insert_fields, insert_values = process_mapping_table(context, cursor_query, insert_fields, insert_values)
      except Exception as mapping_error:
          plpy.error(f"Fehler beim Verarbeiten der Kopfdaten-Mapping-Tabelle: {str(mapping_error)}")

      # SQL-Insert-Anweisung zusammenbauen und ausführen
      try:
          query = f"INSERT INTO eingrechdokument ({', '.join(insert_fields)}) VALUES ({', '.join(insert_values)}) RETURNING beld_id, beld_dokunr, dbrid"
          beld_id = plpy.execute(query)
          if not beld_id:
              plpy.error("Fehler: Kein Ergebnis vom INSERT in 'eingrechdokument' erhalten.")
          beld_id_value = beld_id[0]['beld_id']
          beld_dokunr_value = beld_id[0]['beld_dokunr']
          beld_dbrid_value = beld_id[0]['dbrid']
      except Exception as insert_error:
          plpy.error(f"Fehler beim Einfügen der Kopfdaten in 'eingrechdokument': {str(insert_error)}")

      ###################################
      # belegbezogene Ab- und Zuschläge #
      ###################################

      if hasattr(doc.trade.settlement, 'allowance_charge') and hasattr(doc.trade.settlement.allowance_charge, 'children'):
          for abzu_beleg in doc.trade.settlement.allowance_charge.children:
              # aktuellen Ab/Zuschlag dem Dictionary hinzufügen
              context['abzu_beleg'] = abzu_beleg

              abzu_fields = ['belaz_dokument_id']
              abzu_values = ["'" + str(beld_id_value) + "'"]

              # Auszulesende Felder für belegbezogene Ab- und Zuschläge aus Mappingtabelle holen
              cursor_query = f"""
              SELECT * FROM (
                  SELECT *,
                          ROW_NUMBER() OVER (PARTITION BY erechmap_table, erechmap_column
                          ORDER BY CASE WHEN erechmap_ad_krz = '{beld_krzlieferung}' THEN 0 ELSE 1 END) AS rn
                  FROM x_10_interfaces.e_rechnung_mapping
                  WHERE erechmap_table = 'belegabzu'
                      AND erechmap_doktype = '{doktype}'
                      AND (erechmap_ad_krz = '{beld_krzlieferung}' OR erechmap_ad_krz IS NULL)
                  ) sub
                  WHERE rn = 1
              """

              # Verarbeitung der Ab-/Zuschlagsposition
              abzu_fields, abzu_values = process_mapping_table(context, cursor_query, abzu_fields, abzu_values)

              # SQL-Insert-Anweisung für Ab-/Zuschlagsposition zusammenbauen und ausführen
              try:
                  abzu_query = f"INSERT INTO belegabzu ({', '.join(abzu_fields)}) VALUES ({', '.join(abzu_values)}) RETURNING belaz_id"
                  plpy.notice(abzu_beleg.trade_tax.children[0].rate_applicable_percent._value)
                  plpy.execute(abzu_query)
              except Exception as abzu_error:
                  plpy.error(f"Fehler beim Anlegen der belegbezogenen Ab-/Zuschläge, Statement: {str(abzu_query)}, Fehler: {str(abzu_error)}")


      #######################
      # Rechnungspositionen #
      #######################

      try:
          for item in doc.trade.items.children:
              # aktuelle Position dem Dictionary hinzufügen
              context['item'] = item

              position_fields = ['belp_dokument_id']
              position_values = ["'" + str(beld_id_value) + "'"]

              # Auszulesende Felder für Positionsdaten aus Mappingtabelle holen
              cursor_query = f"""
                SELECT * FROM (
                  SELECT *,
                         ROW_NUMBER() OVER (PARTITION BY erechmap_table, erechmap_column
                           ORDER BY CASE WHEN erechmap_ad_krz = '{beld_krzlieferung}' THEN 0 ELSE 1 END) AS rn
                    FROM x_10_interfaces.e_rechnung_mapping
                   WHERE erechmap_table = 'belegpos'
                     AND erechmap_doktype = '{doktype}'
                     AND (erechmap_ad_krz = '{beld_krzlieferung}' OR erechmap_ad_krz IS NULL)
                  ) sub
                  WHERE rn = 1
                """

              # Verarbeitung der Rechnungsposition
              position_fields, position_values = process_mapping_table(context, cursor_query, position_fields, position_values)

              # SQL-Insert-Anweisung für Position zusammenbauen und ausführen
              try:
                  position_query = f"INSERT INTO belegpos ({', '.join(position_fields)}) VALUES ({', '.join(position_values)}) RETURNING belp_id"
                  belp_id = plpy.execute(position_query)
                  belp_id_value = belp_id[0]['belp_id']
              except Exception as positions_error:
                  plpy.error(f"Fehler beim Anlegen der Rechnungspositionen, Statement: {str(position_query)}, Fehler: {str(positions_error)}")

              #######################################
              # positionsbezogene Ab- und Zuschläge #
              #######################################
              if hasattr(item.settlement, 'allowance_charge') and hasattr(item.settlement.allowance_charge, 'children'):
                  for abzu_pos in item.settlement.allowance_charge.children:
                      # aktuellen Ab/Zuschlag dem Dictionary hinzufügen
                      context['abzu_pos'] = abzu_pos

                      abzu_fields = ['belpaz_belegpos_id']
                      abzu_values = ["'" + str(belp_id_value) + "'"]

                      # Auszulesende Felder für Ab- und Zuschläge aus Mappingtabelle holen
                      cursor_query = f"""
                        SELECT * FROM (
                          SELECT *,
                                 ROW_NUMBER() OVER (PARTITION BY erechmap_table, erechmap_column
                                   ORDER BY CASE WHEN erechmap_ad_krz = '{beld_krzlieferung}' THEN 0 ELSE 1 END) AS rn
                            FROM x_10_interfaces.e_rechnung_mapping
                           WHERE erechmap_table = 'belegposabzu'
                             AND erechmap_doktype = '{doktype}'
                             AND (erechmap_ad_krz = '{beld_krzlieferung}' OR erechmap_ad_krz IS NULL)
                          ) sub
                          WHERE rn = 1
                        """

                      # Verarbeitung der Ab-/Zuschlagsposition
                      abzu_fields, abzu_values = process_mapping_table(context, cursor_query, abzu_fields, abzu_values)

                      # SQL-Insert-Anweisung für Ab-/Zuschlagsposition zusammenbauen und ausführen
                      try:
                          abzu_query = f"INSERT INTO belegposabzu ({', '.join(abzu_fields)}) VALUES ({', '.join(abzu_values)}) RETURNING belpaz_id"
                          plpy.execute(abzu_query)
                      except Exception as abzu_error:
                          plpy.error(f"Fehler beim Anlegen der Ab-/Zuschläge, Statement: {str(abzu_query)}, Fehler: {str(abzu_error)}")
      except Exception as positions_error:
          plpy.error(f"Fehler beim Verarbeiten der Rechnungspositionen, Fehler: {str(positions_error)}")


      # Belegnummer der angelegten Rechnung zurückgeben
      dokunr = str(beld_dokunr_value)
      pd_dbrid = beld_dbrid_value

      # Ablage des Eingangsrechnungsdokumentes in der Datenbank
      query = "SELECT * FROM tsystem.picndoku__delayed_upload__create( _file_bytea => $1, _file_name => $2, _tablename => 'eingrechdokument', _dbrid => $3, _doktype => 'werech' )"
      plan = plpy.prepare( query, ["bytea", "varchar", "varchar"] )
      delayed_upload = plpy.execute( plan, [xml_file, dokunr + ".xml", pd_dbrid] )

      return ( dokunr, pd_dbrid )

    $$ LANGUAGE plpython3u;
--

-- XRechnungs XML aus ZUGFeRD PDF extrahieren
CREATE OR REPLACE FUNCTION x_10_interfaces.xrechnung__import__from_pdf_bytea(
    IN pdf_file bytea,
    IN doktype  varchar DEFAULT 'xrechnung',
    OUT dokunr  varchar,
    OUT dbrid   varchar
  )
  RETURNS record
  AS $$
    import re
    import sys

    # Relativer Pfad zu Modulen (ausgehend vom Arbeitsverzeichnis 'data')
    relative_path = '../python-plpythonu/Lib/site-packages'
    # Pfad nur hinzufügen, wenn er noch nicht in sys.path ist
    if relative_path not in sys.path:
      sys.path.append(relative_path)

    from pypdf import PdfReader
    from io import BytesIO

    def extract_zugferd_xml_with_pypdf(pdf_bytes):
        """Extrahiert eingebettete XML-Datei aus ZUGFeRD-PDF mithilfe von PyPDF."""
        pdf_reader = PdfReader(BytesIO(pdf_bytes))

        # Suche nach eingebetteten Dateien im PDF
        embedded_files = pdf_reader.trailer["/Root"]["/Names"].get("/EmbeddedFiles")
        if embedded_files:
            files = embedded_files["/Names"]
            for i in range(1, len(files), 2):
                file_spec = files[i]
                file_dict = file_spec.get_object()
                file_name = file_dict.get("/F")  # Hole den Dateinamen

                if file_name.endswith(".xml"):
                    # Extrahiere die XML-Daten
                    file_stream = file_dict["/EF"]["/F"].get_object()
                    xml_data = file_stream.get_data()
                    return xml_data.decode('utf-8')

        plpy.error("Keine eingebettete XML-Datei im PDF gefunden")

    # Eingabedaten prüfen
    if pdf_file is None:
        raise ValueError("Es wurde kein PDF-Dokument übergeben.")

    # Versuche das XML aus der PDF zu extrahieren
    xml_data = extract_zugferd_xml_with_pypdf(pdf_file)

    # Bereite die SQL-Abfrage vor und führe sie aus
    query = "SELECT * FROM x_10_interfaces.xrechnung__import__from_xml_bytea($1, $2)"
    plan = plpy.prepare(query, ["bytea", "varchar"])
    result = plpy.execute(plan, [xml_data.encode('utf-8'), doktype])

    # Gib das Ergebnis zurück
    if result:
        dokunr = result[0]["dokunr"]
        pd_dbrid = result[0]["dbrid"]

        # Ablage des Eingangsrechnungsdokumentes in der Datenbank
        query = "SELECT * FROM tsystem.picndoku__delayed_upload__create( _file_bytea => $1, _file_name => $2, _tablename => 'eingrechdokument', _dbrid => $3, _doktype => 'werech' )"
        plan = plpy.prepare( query, ["bytea", "varchar", "varchar"] )
        delayed_upload = plpy.execute( plan, [pdf_file, dokunr + ".pdf", pd_dbrid] )

        return ( dokunr, pd_dbrid )
    else:
        plpy.error("Kein Ergebnis von der Importfunktion zurückgegeben")

  $$ LANGUAGE plpython3u;
--

-- XRechnung über gegebenen Pfad importieren
-- Datei aus Pfad auslesen und an die entsprechende Importfunktion (PDF oder XML) übergeben.
CREATE OR REPLACE FUNCTION x_10_interfaces.xrechnung__import__from_file(
      filepath  varchar,
      doktype   varchar DEFAULT 'xrechnung'
  )
  RETURNS varchar
  AS $$

    try:
        import os

        # Überprüfen, ob der Pfad existiert
        if not os.path.exists(filepath):
            plpy.error(f'Die angegebene Datei existiert nicht: {path}')

        # Überprüfen, ob es sich um eine Datei handelt
        if not os.path.isfile(filepath):
            plpy.error(f'Der angegebene Pfad ist keine Datei: {path}')

        # Datei im Binary-Modus öffnen und lesen
        with open(filepath, "rb") as file:
            file_content = file.read()

        # Dateityp anhand der Dateierweiterung bestimmen
        file_extension = os.path.splitext(filepath)

        # Wenn es sich um eine XML-Datei handelt
        if file_extension.lower() == '.xml':
            # Definieren der SQL-Abfrage für den XML-Import
            query = "SELECT * FROM x_10_interfaces.xrechnung__import__from_xml_bytea($1, $2)"
            plan = plpy.prepare(query, ["bytea", "varchar"])
            rv = plpy.execute(plan, [file_content, doktype])

        # Wenn es sich um eine PDF-Datei handelt
        elif file_extension.lower() == '.pdf':
            # Definieren der SQL-Abfrage für den PDF-Import
            query = "SELECT * FROM x_10_interfaces.xrechnung__import__from_pdf_bytea($1, $2)"
            plan = plpy.prepare(query, ["bytea", "varchar"])
            rv = plpy.execute(plan, [file_content, doktype])

        # Falls ein nicht unterstützter Dateityp vorliegt
        else:
            plpy.error(f'Dateityp {file_extension} wird nicht unterstützt. Nur XML oder PDF ist erlaubt.')

        # Rückgabe des Ergebnisses der aufgerufenen Funktion
        if rv[0]["dokunr"]:
            return rv[0]["dokunr"]
        else:
            plpy.error('Kein Ergebnis von der Importfunktion zurückgegeben')

    except Exception as e:
        plpy.error(f'Fehler beim Importieren der E-Rechnung von Pfad: {str(e)}')

  $$ LANGUAGE plpython3u;

--
CREATE OR REPLACE FUNCTION x_10_interfaces.xrechnung__export_cii__to_xml(
    dokunr     varchar,
    file_path  varchar DEFAULT null -- z.B: 'Y:\\_Archiv_n_Trash\\Trash\\output.xml'
  )
  RETURNS bytea
  AS $$

    import os
    import sys

    # Relativer Pfad zu Modulen (ausgehend vom Arbeitsverzeichnis 'data')
    relative_path = '../python-plpythonu/Lib/site-packages'
    # Pfad nur hinzufügen, wenn er noch nicht in sys.path ist
    if relative_path not in sys.path:
      sys.path.append(relative_path)

    # plpy.notice(sys.path)

    from datetime import date, datetime, timezone
    from decimal  import Decimal, ROUND_HALF_UP
    if sys.version_info >= (3, 8):
        from importlib.metadata import version
    else:
        from importlib_metadata import version

    from drafthorse.models.accounting import ApplicableTradeTax, TradeAllowanceCharge, CategoryTradeTax
    from drafthorse.models.document   import Document
    from drafthorse.models.note       import IncludedNote
    from drafthorse.models.tradelines import LineItem
    from drafthorse.models.payment    import PaymentTerms
    from drafthorse.models.party      import TaxRegistration, URIUniversalCommunication
    from drafthorse.models            import NS_RAM
    from drafthorse.models.elements   import Element


    def kaufm_runden(wert, stellen = 2):
      """
      Kaufmännisches Runden des Wertes (es wird bei 5 immer aufgerundet)
      """
      # Format für die Quantisierung, z.B. 2 Stellen nach dem Komma: '0.01'
      form = Decimal('1e-{}'.format(stellen)) if stellen > 0 else Decimal('1')
      return Decimal(wert).quantize(form, rounding=ROUND_HALF_UP)

    def beleg_typ_to_xml_string(typ: str = None) -> str:
      """
      Convert a type string to its XML equivalent based on the first character.
      :param typ: The type string indicating the type of invoice.
      :return: The XML equivalent string for the invoice type.
      """
      # Set default character if typ is empty
      t = typ[0] if typ else 'R'

      # Map the character to the corresponding XML string
      if t == 'A':
          return '386'  # Prepayment invoice
      elif t == 'G':
          return '383'  # Debit note (interpreted as negative charge)
      elif t == 'T':
          return '326'  # Partial invoice
      else:
          return '380'  # Default: Commercial invoice

    #def tax_code_get(pos_steuproz: float = None) -> str:   # => Fuktion wird nicht mehr benötigt, da nun der Steuercode direkt übergeben wird:
    #  """
    #  Gibt den Steuer-Code ('G' oder 'S') basierend auf dem Wert von pos_steuproz zurück.
    #  Args:
    #      pos_steuproz (int or float): Der Steuerprozentsatz.
    #  Returns:
    #      str: 'G' wenn pos_steuproz == 0, sonst 'S'.
    #  """
    #  if pos_steuproz is None:
    #    raise ValueError("pos_steuproz darf nicht None sein.")
    #  return 'G' if pos_steuproz == 0 else 'S'


    # Eingabedaten prüfen
    if dokunr is None:
      raise ValueError("Es wurde keine Rechnungsnummer angegeben.")

    # Standard-SQLs abrufen
    SSQL = {
      "Kopf"           : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.Kopf')")[0]["systemsqlstatement__query"],
      "Konto"          : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.Konto')")[0]["systemsqlstatement__query"],
      "Pos"            : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.Pos')")[0]["systemsqlstatement__query"],
      "Auftag"         : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.Auftrag')")[0]["systemsqlstatement__query"],
      "EigenAdresse"   : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.EigenAdresse')")[0]["systemsqlstatement__query"],
      "SteuerTeilsumme": plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.SteuerTeilsumme')")[0]["systemsqlstatement__query"],
      "LieferAdresse"  : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.LieferAdresse')")[0]["systemsqlstatement__query"],
      "AbZuGlobal"     : plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.PosAbZuGlobal')")[0]["systemsqlstatement__query"]
    }

    # Standard-SQL (Kopf) ausführen
    invoice_data = {}
    for key, query in SSQL.items():
      if key == "EigenAdresse":
        plan = plpy.prepare(query.replace(":kopf_ansprechpartner", "$1"), ["text"])
        # plpy.notice(key + " " + query.replace(":kopf_ansprechpartner", "$1"))
        result = plpy.execute(plan, [invoice_data["Kopf"][0]["d_apint"]])
      else:
        plan = plpy.prepare(query.replace(":dokunr", "$1"), ["text"])
        # plpy.notice(key + " " + query.replace(":dokunr", "$1"))
        result = plpy.execute(plan, [dokunr])
      if result:
        invoice_data[key] = result

    ###################
    # Beleg erstellen #
    ###################

    doc = Document()
    # Standard-XRechnung 3.0 (CIUS) OHNE Extension
    doc.context.guideline_parameter.id = "urn:cen.eu:en16931:2017#compliant#urn:xeinkauf.de:kosit:xrechnung_3.0"
    # Peppol-Process-ID
    doc.context.business_parameter.id = "urn:fdc:peppol.eu:2017:poacc:billing:01:1.0"

    #######################
    # Kopfdaten erstellen #
    #######################

    doc.header.id                                = dokunr                                                                        # (BT-1)   Rechnungsnummer (Eindeutige ID)
    doc.header.type_code                         = beleg_typ_to_xml_string(invoice_data["Kopf"][0]["d_typ"])                     # (BT-8)   Rechnungstyp (Code)
                                                                                                                                 # (BT-10)  Leitweg-ID
    doc.header.issue_date_time                   = datetime.strptime(invoice_data["Kopf"][0]["d_datum_erfasst"], "%Y-%m-%d")     # (BT-2)   Rechnungsdatum
    doc.trade.delivery.event.occurrence          = datetime.strptime(invoice_data["Kopf"][0]["beld_erstelldatum"], "%Y-%m-%d")   # (BT-72)  Lieferdatum

    # Dokument-Referenzen
    doc.trade.delivery.despatch_advice.issuer_assigned_id = invoice_data["Kopf"][0]["bz_lfs"]                                    # (BT-16)  Lieferscheinnummer des Verkäufers (wir)
    doc.trade.agreement.seller_order.issuer_assigned_id   = invoice_data["Auftag"][0]["ag_nr"]                                   # (BT-14)  Auftragsnummer des Verkäufers (wir) /* TODO: Hier darf die Auftragsnummer nur gefüllt werden wenn diese auch übergeben wurde */

    #doc.header.languages.add("de")
    #doc.header.name = dokunr

    # Kopftext
    if invoice_data["Kopf"][0]["d_kopftext_txt"]:
      note = IncludedNote()
      note.content.add(invoice_data["Kopf"][0]["d_kopftext_txt"])                                                                # (BT-22)  Notiz-Text
      #note.content_code = "AAI"                                                                                                 # (BT-21)  Notiz-Code (AAI=Allgemeine Informationen) (UNTDID 4451)
      doc.header.notes.add(note)
    # Fußtext
    if invoice_data["Kopf"][0]["d_fusstext_txt"]:
      note = IncludedNote()
      note.content.add(invoice_data["Kopf"][0]["d_fusstext_txt"])                                                                # (BT-22)  Notiz-Text
      #note.content_code = "AAI"                                                                                                 # (BT-21)  Notiz-Code (AAI=Allgemeine Informationen) (UNTDID 4451)
      doc.header.notes.add(note)

    #################################
    # Erstellen der Handelsparteien #
    #################################
    # Verkäufer (wir) #
    ###################

    if invoice_data["EigenAdresse"][0]["ad_fa1"]:
      # Adresse (wir)
      doc.trade.agreement.seller.name                      = invoice_data["EigenAdresse"][0]["ad_fa1"]                           # (BT-27)  Name/Firma des Verkäufers
      doc.trade.agreement.seller.address.country_id        = invoice_data["EigenAdresse"][0]["ad_landiso"]                       # (BT-40)  Land des Verkäufers
      doc.trade.agreement.seller.address.city_name         = invoice_data["EigenAdresse"][0]["ad_ort"]                           # (BT-37)  Stadt des Verkäufers
      doc.trade.agreement.seller.address.postcode          = invoice_data["EigenAdresse"][0]["ad_plz"]                           # (BT-38)  PLZ des Verkäufers
      doc.trade.agreement.seller.address.line_one          = invoice_data["EigenAdresse"][0]["ad_str"]                           # (BT-35)  Straße des Verkäufers
      if version('drafthorse') == '2.4.0':                                                                                       # (BT-34)  Elektronische Adresse des Verkäufers
        doc.trade.agreement.seller.electronic_address.add(URIUniversalCommunication( \
                                                                    uri_ID=("EM", invoice_data["EigenAdresse"][0]["ap_mail"])))
      else: # NEU: ab drafthorse >= '2025.1.0'
        doc.trade.agreement.seller.electronic_address.uri_ID = ("EM", invoice_data["EigenAdresse"][0]["ap_mail"])

      doc.trade.agreement.seller.id                        = str(invoice_data["Kopf"][0]["a1_eknr"])                             # (BT-29)  eigene Lieferantennummer beim Kunden

      # Steuerkennung (wir)
      doc.trade.agreement.seller.tax_registrations.add(TaxRegistration( \
                                                                    id=("VA", invoice_data["EigenAdresse"][0]["ad_ustidnr"])))   # (BT-31)  Umsatzsteuer-ID des Verkäufers
      #doc.trade.agreement.seller.tax_registrations.add(TaxRegistration(id=(...tbd...)))                                         # (BT-32)  Steuer-Nr. des Verkäufers

      # Ansprechpartner (wir)
      ap_vorn = invoice_data["EigenAdresse"][0]["ap_vorn"]
      ap_name = invoice_data["EigenAdresse"][0]["ap_name"]
      doc.trade.agreement.seller.contact.person_name      = " ".join( filter( None, [ap_vorn, ap_name] ) )                       # (BT-41)  Ansprechpartner des Verkäufers (Kontaktperson)
      doc.trade.agreement.seller.contact.department_name  = ""
      doc.trade.agreement.seller.contact.telephone.number = invoice_data["EigenAdresse"][0]["ap_tel"]                            # (BT-42)  Telefonnummer des Verkäufers (Kontaktperson)
      doc.trade.agreement.seller.contact.email.address    = invoice_data["EigenAdresse"][0]["ap_mail"]                           # (BT-43)  E-Mail-Adresse des Verkäufers (Kontaktperson)
    else:
      plpy.notice("Informationen in der eigenen Adresse fehlen (Firmenname). X-Rechnungserstellung wird abgebrochen.")
      plpy.execute("SELECT PRODAT_ERROR(35344)")                                                                                 # Es kann keine X-Rechnung erstellt werden wenn kein Firmenname in der Adresse # definiert ist.
      return None

    ##################
    # Käufer (Kunde) #
    ##################

    # Adresse (Kunde)
    doc.trade.agreement.buyer.name                      = invoice_data["Kopf"][0]["ad_fa1"]                                      # (BT-44)  Name/Firma des Käufers
    doc.trade.agreement.buyer.address.country_id        = invoice_data["Kopf"][0]["ad_landiso"]                                  # (BT-55)  Land des Käufers
    doc.trade.agreement.buyer.address.city_name         = invoice_data["Kopf"][0]["ad_ort"]                                      # (BT-52)  Stadt des Käufers
    doc.trade.agreement.buyer.address.postcode          = invoice_data["Kopf"][0]["ad_plz"]                                      # (BT-53)  PLZ des Käufers
    doc.trade.agreement.buyer.address.line_one          = invoice_data["Kopf"][0]["ad_str"]                                      # (BT-50)  Straße des Käufers

    if version('drafthorse') == '2.4.0':                                                                                         # (BT-49)  Elektronische Adresse des Käufers
      doc.trade.agreement.buyer.electronic_address.add(URIUniversalCommunication( \
                                                                   uri_ID=("EM", invoice_data["Kopf"][0]["ad_email1"])))
    else: # NEU: ab drafthorse >= '2025.1.0'
      doc.trade.agreement.buyer.electronic_address.uri_ID = ("EM", invoice_data["Kopf"][0]["ad_email1"])

    doc.trade.agreement.buyer.id                        = str(invoice_data["Kopf"][0]["a1_knr"])                                 # (BT-46)  Kundennummer/Deditorennummer

    # Referenz (Kunde)
    doc.trade.agreement.buyer_reference = invoice_data["Kopf"][0]["a1_leitweg_id"]                                               # (BT-10)  Leitweg-ID

    # Steuerkennung (Kunde)
    doc.trade.agreement.buyer.tax_registrations.add(TaxRegistration(id=("VA", invoice_data["Kopf"][0]["d_ad_ustidnr"])))         # (BT-48)  Umsatzsteuer-ID des Käufers

    # Ansprechpartner (Kunde)
    ap_vorn = invoice_data["Kopf"][0]["ap_vorn"]
    ap_name = invoice_data["Kopf"][0]["ap_name"]
    doc.trade.agreement.buyer.contact.person_name      = " ".join( filter( None, [ap_vorn, ap_name] ) )                          # (BT-56)  Ansprechpartner des Käufers (Kontaktperson)
    doc.trade.agreement.buyer.contact.department_name  = ""
    doc.trade.agreement.buyer.contact.telephone.number = invoice_data["Kopf"][0]["ap_tel"]                                       # (BT-57)  Telefonnummer des Käufers (Kontaktperson)
    doc.trade.agreement.buyer.contact.email.address    = invoice_data["Kopf"][0]["ad_email1"]                                    # (BT-58)  E-Mail-Adresse des Käufers (Kontaktperson)

    #########################
    # Lieferadresse (Kunde) #
    #########################

    doc.trade.delivery.ship_to.name               = invoice_data["LieferAdresse"][0]["ad_fa1"]                                   # (BT-70)  Name/Firma der Lieferadresse
    doc.trade.delivery.ship_to.address.country_id = invoice_data["LieferAdresse"][0]["ad_landiso"]                               # (BT-80)  Land der Lieferadresse
    doc.trade.delivery.ship_to.address.city_name  = invoice_data["LieferAdresse"][0]["ad_ort"]                                   # (BT-77)  Stadt der Lieferadresse
    doc.trade.delivery.ship_to.address.postcode   = invoice_data["LieferAdresse"][0]["ad_plz"]                                   # (BT-78)  PLZ der Lieferadresse
    doc.trade.delivery.ship_to.address.line_one   = invoice_data["LieferAdresse"][0]["ad_str"]                                   # (BT-75)  Straße der Lieferadresse

    ########################
    # Zahlungsvereinbarung #
    ########################

    #doc.trade.settlement.payee.name   = ...tbd...                                                                               # (BT-59)  abweichender Zahlungsempfänger
    doc.trade.settlement.currency_code = invoice_data["Kopf"][0]["d_waco"]                                                       # (BT-5)   Währung

    # Zahlungsempfängerkonto und Bank
    if "Konto" in invoice_data:
      doc.trade.settlement.payment_means.type_code             = "ZZZ"                                                           # (BT-82)  Code: Zahlungsmittel (UN/ECE 4461)
      doc.trade.settlement.payment_means.payee_account.iban    = invoice_data["Konto"][0]["kto_iban"]                            # (BT-84)  IBAN
      doc.trade.settlement.payment_means.payee_institution.bic = invoice_data["Konto"][0]["kto_bic"]                             # (BT-86)  BIC
    else:
      plpy.notice("Es gibt keine Kontoverbindung. X-Rechnungserstellung wird abgebrochen.")
      plpy.execute("SELECT PRODAT_ERROR(35343)")                                                                                 # Es kann keine X-Rechnung zu Rechnungen ohne Kontoverbindung erstellt werden.
      return None

    # Zahlungsbedingungen
    payment_terms = PaymentTerms()
    payment_terms.due         = datetime.strptime(invoice_data["Kopf"][0]["d_datum_soll"], "%Y-%m-%d")                           # (BT-9)   Fälligkeitstermin
    if invoice_data["Kopf"][0]["d_zahlung_sks"] > 0:                                                                             # falls Skonto gewährt wird
      tage = invoice_data["Kopf"][0]["d_zahlung_skv"]
      proz = invoice_data["Kopf"][0]["d_zahlung_sks"]
      payment_terms.description = f"#SKONTO#TAGE={tage}#PROZENT={proz}#\n"                                                       # (BT-20)  Zahlungsbedingungen mit Skonto
    doc.trade.settlement.terms.add(payment_terms)

    #######################
    # Rechnungspositionen #
    #######################

    if "Pos" in invoice_data:
      for Beleg_Position in invoice_data["Pos"]:

        li = LineItem()
        li.document.line_id = str(Beleg_Position["p_pos"])                                                                       # (BT-126) Positionsnummer

        # Artikel

        li.product.name               = Beleg_Position["p_akbez"]                                                                # (BT-153) Artikelbezeichnung
        li.product.description        = Beleg_Position["ak_txt"]                                                                 # (BT-154) Artikelbeschreibung
        li.product.seller_assigned_id = Beleg_Position["p_aknr"]                                                                 # (BT-152) Artikelnummer des Verkäufers
        li.product.buyer_assigned_id  = Beleg_Position["p_aknr_referenz"]                                                        # (BT-155) Artikelnummer des Käufers

        # Bestellreferenz des Kunden
        doc.trade.agreement.buyer_order.issuer_assigned_id = ",".join( set( filter( None, [doc.trade.agreement.buyer_order.issuer_assigned_id._text, Beleg_Position["p_refnummer"] ] ) ) )     # (BT-13)  Bestellreferenz des Kunden (Bestellnummer)
        li.agreement.buyer_order.issuer_assigned_id = Beleg_Position["p_refpos"]                                                 # (BT-132) Bestellpositionsreferenz des Kunden

        # Preis
        #li.agreement.gross.amount         = ...tbd...
        #li.agreement.gross.basis_quantity = (Decimal(Beleg_Position["p_menge"]), Beleg_Position["me_unitcode"])
        #li.agreement.net.amount           = Decimal(Beleg_Position["p_preis_me"])                                               # (BT-146) Netto-Preis der Position
        li.agreement.net.amount           = Decimal(Beleg_Position["p_preis_netto"])                                             # (BT-146) Netto-Preis der Position inkl. Pos.-Rabatt
        li.agreement.net.basis_quantity   = (Decimal(Beleg_Position["p_preiseinheit"]), Beleg_Position["me_unitcode"])           # (BT-149) Preiseinheit mit Mengeneinheit
        li.delivery.billed_quantity       = (Decimal(Beleg_Position["p_menge"]), Beleg_Position["me_unitcode"])                  # (BT-129) verrechnete Anzahl mit Mengeneinheit

        # Steuer
        li.settlement.trade_tax.type_code               = "VAT"
        li.settlement.trade_tax.category_code           = Beleg_Position["p_steu_untdid"]                                        # (BT-151) Code für die Umsatzsteuerkategorie, UNTDID 5305 (G / S)
        li.settlement.trade_tax.rate_applicable_percent = Decimal(Beleg_Position["p_steu_proz"])                                 # (BT-152) Umsatzsteuersatz

        # Positionswert
        li.settlement.monetary_summation.total_amount   = kaufm_runden(Decimal(Beleg_Position["p_wert_tot_netto"]), 2)           # (BT-131) Netto-Gesamtwert der Position

        # Positions Ab- und Zuschläge

        query = plpy.execute("SELECT TSystem.systemsqlstatement__query('TWawi.XRechnung.Generate.PosAbZuPos')")[0]["systemsqlstatement__query"]
        plan = plpy.prepare(query.replace(":dokunr", "$1").replace(":pos", "$2"), ["text", "int"])
        AbZuPos = plpy.execute(plan, [dokunr, Beleg_Position["p_pos"]])

        for AbZu_einzeln in AbZuPos:

          # Ab-/Zuschlag
          pos_allowance = TradeAllowanceCharge()
          pos_allowance.indicator     = Decimal(AbZu_einzeln["beaz_tot"]) >= 0                                                   #          Typ (False = Abschlag, True = Zuschlag)
          pos_allowance.actual_amount = kaufm_runden(Decimal(abs(AbZu_einzeln["beaz_tot"])), 2)                                  # (BT-136)/(BT-141) Netto-Betrag des Ab-/Zuschlags
                                                                                                                                 # (BT-137)/(BT-142) Grundbetrag des Ab-/Zuschlags
                                                                                                                                 # (BT-138)/(BT-143) Prozentwert des Ab-/Zuschlags
          pos_allowance.reason        = AbZu_einzeln["abz_txt"]                                                                  # (BT-139)/(BT-144) Grund (Beschreibung)
          #pos_allowance.reason_code  = ...tbd...                                                                                # (BT-140)/(BT-145) Grund (Code) Codelisten für Ab/Zuschläge: UNTDID 5189 / UNTDID 7161

          # Steuern auf AbZuSchlag => positionsbezoge Ab/Zuschläge haben keine gesonderten Steuerinformationen
          #pos_allowance.trade_tax.rate_applicable_percent = Decimal(AbZu_einzeln["az_sproz"])                                    #
          #pos_allowance.trade_tax.type_code               = "VAT"
          #pos_allowance.trade_tax.category_code           = tax_code_get(Decimal(AbZu_einzeln["az_sproz"]))                      #          G / S

          # Hinzufügen zur Positionsebene
          li.settlement.allowance_charge.add(pos_allowance)

        doc.trade.items.add(li)
    else:
      plpy.notice("Es gibt keine Belegpositionen. X-Rechnungserstellung wird abgebrochen.")
      plpy.execute("SELECT PRODAT_ERROR(35342)")                                                                                 # Es kann keine X-Rechnung zu Rechnungen ohne Positionen erstellt werden
      return None

    ###########################
    # Beleg Ab- und Zuschläge #
    ###########################

    if "AbZuGlobal" in invoice_data:
      for AbZu_einzeln in invoice_data["AbZuGlobal"]:

        # Ab-/Zuschlag
        document_allowance = TradeAllowanceCharge()
        document_allowance.indicator     = Decimal(AbZu_einzeln["beaz_tot"]) >= 0                                                #          Typ (False = Abschlag, True = Zuschlag)
        document_allowance.actual_amount = kaufm_runden(Decimal(abs(AbZu_einzeln["beaz_tot"])), 2)                               # (BT-92)/(BT-99)  Netto-Betrag des Ab-/Zuschlags
        document_allowance.reason        = AbZu_einzeln["abz_txt"]                                                               # (BT-97)/(BT-104) Grund (Beschreibung)
        #document_allowance.reason_code  = ...tbd...                                                                             # (BT-98)/(BT-105) Grund (Code) Codelisten für Ab-/Zuschläge: UNTDID 5189 / UNTDID 7161

        # Steuern auf AbZuSchlag
        tax = CategoryTradeTax()
        tax.rate_applicable_percent = Decimal(AbZu_einzeln["az_sproz"])                                                          # (BT-96)/(BT-103) Umsatzsteuersatz des Ab-/Zuschlags
        tax.type_code               = "VAT"
        tax.category_code           = AbZu_einzeln["az_steu_untdid"]                                                             # (BT-95)/(BT-102) Code für Umsatzsteuerkategorie des Ab-/Zuschlags, UNTDID 5305 (G / S)

        # Steuer dem Ab-/Zuschlag hinzufügen
        document_allowance.trade_tax.add(tax)

        # Hinzufügen zur Dokumentebene
        doc.trade.settlement.allowance_charge.add(document_allowance)
    else:
      plpy.notice("Beleg Ab- und Zuschläge konnte nicht geladen werden. wird übersprungen.")


    ################
    # Belegsteuern #
    ################

    if "SteuerTeilsumme" in invoice_data:
      for steuerzeile in invoice_data["SteuerTeilsumme"]:
        trade_tax = ApplicableTradeTax()
        trade_tax.calculated_amount       = kaufm_runden(Decimal(steuerzeile["p_sum_steuer"]), 2)                                # (BT-117) Umsatzsteuerwert
        trade_tax.basis_amount            = kaufm_runden(Decimal(steuerzeile["p_sum_netto"]), 2)                                 # (BT-116) Gesamt-Netto inkl. Beleg AbZu
        trade_tax.type_code               = "VAT"                                                                                #
        trade_tax.category_code           = steuerzeile["p_steu_untdid"]                                                         # (BT-118) Code für Umsatzsteuerkategorie, UNTDID 5305 (G / S)
        trade_tax.rate_applicable_percent = Decimal(steuerzeile["p_steu_proz"])                                                  # (BT-119) Umsatzsteuersatz
        #trade_tax.exemption_reason_code = 'VATEX-EU-AE' (tbd: steuerfreie Positionen)                                           # (BT-121) Code für den Grund der Befreiung von der Umsatzsteuer

        # Hinzufügen zur Dokumentebene
        doc.trade.settlement.trade_tax.add(trade_tax)
    else:
      plpy.notice("Es gibt keine Belegsteuern. wird übersprungen.")

    ###############
    # Belegsummen #
    ###############

    # Berechnung von Netto-Summen der Beleg-Ab/Zuschläge
    charge_total = sum(Decimal(charge.actual_amount._value) \
                       for charge in doc.trade.settlement.allowance_charge.children if charge.indicator)
    allowance_total = sum(Decimal(allowance.actual_amount._value) \
                          for allowance in doc.trade.settlement.allowance_charge.children if not allowance.indicator)

    # Berechnung von Netto-Summe der Positionen (!ohne Beleg Ab/Zuschläge!)
    net_positions_total = sum(Decimal(item.settlement.monetary_summation.total_amount._value) \
                              for item in doc.trade.items.children)

    # Berechnung von Steuer-Summe der Positionen
    tax_total = sum(Decimal(trade_tax.calculated_amount._value) \
                              for trade_tax in doc.trade.settlement.trade_tax.children)

    doc.trade.settlement.monetary_summation.charge_total    = kaufm_runden(Decimal(charge_total), 2)                                    # (BT-108) Netto-Summe: Alle Zuschläge auf Dokumentebene
    doc.trade.settlement.monetary_summation.allowance_total = kaufm_runden(Decimal(allowance_total), 2)                                 # (BT-107) Netto-Summe: Alle Abschläge auf Dokumentebene
    doc.trade.settlement.monetary_summation.line_total      = kaufm_runden(Decimal(net_positions_total), 2)                             # (BT-106) Netto-Summe: Positionen ohne Beleg-AbZu
    doc.trade.settlement.monetary_summation.tax_basis_total = kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_netto"]), 2)         # (BT-109) Netto-Summe: Gesamtbetrag inkl AbZu
    doc.trade.settlement.monetary_summation.tax_total       = (  kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_brutto"]), 2) \
                                                               - kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_netto"]), 2), \
                                                               invoice_data["Kopf"][0]["d_waco"])                                         # (BT-110) Steuer-Summe: Gesamtbetrag inkl AbZu
    doc.trade.settlement.monetary_summation.grand_total     = kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_brutto"]), 2)        # (BT-112) Gesamtbetrag inkl. Steuern
    doc.trade.settlement.monetary_summation.due_amount      = kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_brutto"]), 2)        # (BT-115) offener Gesamtbetrag inkl. Steuern
    doc.trade.settlement.monetary_summation.rounding_amount =   kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_brutto"]), 2) \
                                                              - kaufm_runden(Decimal(invoice_data["Kopf"][0]["d_wert_tot_netto"]), 2) \
                                                              - kaufm_runden(tax_total, 2)                                              # (BT-114) Betrag, um den der Rechnungsbetrag gerundet wurde

    # Generate XML file
    xml = doc.serialize(schema="FACTUR-X_EXTENDED")

    # Speichern des XML-Inhalts in der angegebenen Datei
    if file_path:
      # Konvertieren der Bytes in einen String
      xml_str = xml.decode("utf-8")
      with open(file_path, "w", encoding="utf-8") as file:
        file.write(xml_str)

    # Return XML
    return xml

  $$ LANGUAGE plpython3u;

CREATE OR REPLACE FUNCTION x_10_interfaces.xrechnung__zugferd_generate__from_pdf_xml_bytea(
    pdf_data bytea,
    xml_data bytea
  )
  RETURNS bytea
  AS $$

    import sys

    # Relativer Pfad zu Modulen (ausgehend vom Arbeitsverzeichnis 'data')
    relative_path = '../python-plpythonu/Lib/site-packages'
    # Pfad nur hinzufügen, wenn er noch nicht in sys.path ist
    if relative_path not in sys.path:
      sys.path.append(relative_path)

    from io import BytesIO
    from drafthorse.pdf import attach_xml

    # Eingabedaten prüfen
    if pdf_data is None:
        raise ValueError("PDF-Daten dürfen nicht NULL sein.")
    if xml_data is None:
        plpy.notice("Es sind keine XML-Daten vorhanden. PDF wird ohne eingebettetes XML zurückgegeben.")
        return pdf_data

    # PDF- und XML-Daten aus bytea lesen
    pdf_bytes = bytes(pdf_data)
    xml_bytes = bytes(xml_data)

    # XML in das PDF einbetten
    pdf_metadata = {
        "Producer": "PRODAT ERP",
        "Creator": "PRODAT ERP"
    }
    zugferd_pdf_bytes = attach_xml(pdf_bytes, xml_bytes, None, pdf_metadata)

    # ZUGFeRD als bytea zurückgeben
    return zugferd_pdf_bytes

  $$ LANGUAGE plpython3u;
